/*
 * - - - - - - - - - - 
 * Paul Spitzer (c)
 * - - - - - - - - - - 
 */
package spitzer
{
	import away3d.core.proto.ObjectContainer3D;
	import away3d.core.proto.Object3D;
	import away3d.core.material.IMaterial;
	import away3d.core.material.WireColorMaterial;
	import away3d.objects.Cube;
	import away3d.objects.WireCube;
	import away3d.core.material.ColorMaterial;
	import away3d.core.material.ISegmentMaterial;
	import away3d.core.material.WireframeMaterial;
	import away3d.core.material.ShadingColorMaterial;
	
	/**
	 * Represents a graph node in 3D. It's really just 
	 * a 3D object that can be shown as empty or a block.
	 */
	public class Node3D extends ObjectContainer3D
	{
		
		public static const SIZE: Number = 80;
		
		// The 3D positions in the graph
		
		public var xPos: uint;
		public var yPos: uint;
		public var zPos: uint;
		
		// Score
		
		public var f: Number;
		public var g: Number = 0;
		public var h: Number;
		
		public var parentNode: Node3D;
		
		//
		
		private var object3D: Object3D
		private var object3DMaterial: IMaterial;

		private var ambient: Number = 0x999999;		
		private var color: Number = 0xcccccc;
		private var alpha: Number = 1;
		
		// State
		
		private var walkable: Boolean = true;
		private var start: Boolean = false;
		private var destination: Boolean = false;
		private var path: Boolean = false;
		
		//
		
		/**
		 * Constructs a new Node3D object.
		 */
		public function Node3D()
		{
			this.update();	
		}
		
		// State modification
		
		/**
		 * Sets the nodes state to either walkable or not
		 */
		public function setWalkable(walkable: Boolean): void
		{
			this.walkable = walkable;
			this.update();
		}
		
		/**
		 * @return Boolean - the Walkable state of the node
		 */
		public function getWalkable(): Boolean
		{
			return this.walkable;
		}
		
		/**
		 * Sets the nodes as the start node
		 */
		public function setStart(start: Boolean): void
		{
			this.start = start;
			this.color = (start) ? 0x22ffaa : 0xcccccc;
			this.update();
		}
		
		/**
		 * Sets the nodes as the destination node
		 */
		public function setDestination(destination: Boolean): void
		{
			this.destination = destination;
			this.color = (destination) ? 0xff4422 : 0xcccccc;
			this.update();
		}
		
		/**
		 * Flags the node as a part of the path
		 */
		public function setPath(path: Boolean): void
		{
			// uncomment to show the path
			this.path = path;
			this.color = (path) ? 0xffffff : 0xcccccc;
			this.ambient = (path) ? 0xffffff : 0x999999;
			this.update();
		}
		
		//
		
		/**
		 * Updates the Node changing it's visual appearance 
		 * to reflect its state
		 */
		private function update(): void
		{
			if(this.object3D != null)
			{
				this.removeChild(this.object3D);
				this.object3D = null;
			}
			
			if(!this.walkable || this.start || this.destination || this.path)
			{
				var s2: Number = SIZE / 2;
				var x: Number = s2;
				var y: Number = s2;
				var z: Number = s2;
				var init: Object = {x: x, y: y, z: z, width: SIZE, height: SIZE, depth: SIZE};
				
				if(!this.walkable)
				{
					this.object3DMaterial = new ShadingColorMaterial(this.ambient, this.color, 0xffffff);
					this.object3D = new Cube(this.object3DMaterial, init);
				}
				else if(this.start || this.destination || this.path)
				{
					this.object3DMaterial = new WireframeMaterial(this.color, 0.2);
					this.object3D = new Cube((this.object3DMaterial as ISegmentMaterial), init);
				}
				
				this.object3D.extra = {node: this};
				this.addChild(this.object3D);
			}
		}
		
		//
		
		/**
		 * @return String - String representation of the object
		 */
		override public function toString(): String
		{
			return "[Node3D " + this.xPos + "," + this.yPos + "," + this.zPos  + " - " + this.g + "," + this.h + "," + this.f + "]";
		}
		
	}
	
}