/*
 * - - - - - - - - - - 
 * Paul Spitzer (c)
 * - - - - - - - - - - 
 */
package
{

	import spitzer.core.Application;
	import away3d.core.proto.Camera3D;
	import away3d.core.proto.Scene3D;
	import away3d.core.proto.View3D;
	import flash.display.SimpleButton;
	import spitzer.Node3D;
	import away3d.core.proto.Object3D;
	import away3d.core.proto.Light3D;
	import away3d.core.proto.ObjectContainer3D;
	import flash.events.KeyboardEvent;
	import flash.ui.Keyboard;
	import flash.events.Event;
	import away3d.core.proto.MouseEvent3D;
	import flash.events.MouseEvent;
	import flash.text.TextFormat;
	import flash.text.TextField;
	import away3d.core.geom.Face3D;
	import away3d.core.draw.DrawTriangle;
	import away3d.core.material.ShadingColorMaterial;
	import away3d.objects.Sphere;
	import caurina.transitions.Tweener;
	
	[SWF(backgroundColor="0x666666")]
	
	/**
	 * Application class for the A* 3D demo
	 */
	public class AStar3D extends Application
	{
		
		// Mode Constants
		
		private static const NONE: uint = 0;
		private static const CREATE_WALL: uint = 1;
		private static const CLEAR_WALL: uint = 2;
		private static const START: uint = 3;
		private static const STOP: uint = 4;
		
		// 3D scene memebers
		
		private var camera: Camera3D;
		private var scene: Scene3D;
		private var view: View3D;
		private var xRotContainer: ObjectContainer3D;
		private var yRotContainer: ObjectContainer3D;
		
		// Node members
		
		private var nodes: Array;
		private var xDepth: uint = 5; // columns
		private var yDepth: uint = 5; // rows
		private var zDepth: uint = 5;
		private var startNode: Node3D;
		private var destinationNode: Node3D;
		
		// Scene Interaction
		
		private var rotatingY: int = 0;
		private var rotatingX: int = 0;
		private var selectedField: TextField;
		private var mode: uint = 0;
		
		// A*
		
		private var path: Array;
		private var open: Array;
		private var closed: Array;
		
		//
		
		/**
		 * Constructs a new AStar3D object.
		 */
		public function AStar3D()
		{
			this.configureStageListeners();
			this.configureLinks();
			this.configureScene();
		}
		
		// Initialization and Configuration
		
		/**
		 * Sets up event listeners on the stage
		 */
		private function configureStageListeners(): void
		{
			this.stage.addEventListener(KeyboardEvent.KEY_DOWN, this.handleKeyDown);
			this.stage.addEventListener(KeyboardEvent.KEY_UP, this.handleKeyUp);
			this.stage.addEventListener(Event.ENTER_FRAME, this.handleEnterFrame);
		}
		
		/**
		 * Sets up the links that provide interaction functionality 
		 * with the nodes and scene
		 */
		private function configureLinks(): void
		{
			var format: TextFormat = new TextFormat();
			format.font = "Verdana";
			format.color = 0x3399ff;
			
			this.configureField(format, "create wall", 5);
			this.configureField(format, "clear wall", 90);
			this.configureField(format, "start node", 165);
			this.configureField(format, "stop node", 250);
			this.configureField(format, "find path", 330);
			this.configureField(format, "clear", 400);
		}
		
		/**
		 * Creates and configures a single link field
		 */
		private function configureField(format: TextFormat, text: String, x: Number): void
		{
			var field: TextField = new TextField();
			field.x = x;
			field.y = 3;
			field.selectable = false;
			field.addEventListener(MouseEvent.CLICK, this.handleFieldClick);
			field.text = text;
			field.setTextFormat(format);
			this.addChild(field);
		}
		
		/**
		 * Creates and sets up the 3D scene
		 */
		private function configureScene(): void
		{
			this.camera = new Camera3D({x: 0, y: 1200, z: 1000, lookat: new Object3D()});
			this.scene = new Scene3D();
			
			this.xRotContainer = new ObjectContainer3D();
			this.yRotContainer = new ObjectContainer3D();
			
			this.xRotContainer.addChild(this.yRotContainer);
			this.scene.addChild(this.xRotContainer);
			
			var light: Light3D = new Light3D(0xffffff, 0.25, 0.25, 0.25);
			light.y = 300;
			light.x = 300
			light.z = -300;
			light.showsource = false;
			this.scene.addChild(new ObjectContainer3D(null, light));
			
			light = new Light3D(0xffffff, 0.25, 0.25, 0.25);
			light.y = 300;
			light.x = -200
			light.z = 300;
			light.showsource = false;
			this.scene.addChild(new ObjectContainer3D(null, light));
			
			this.view = new View3D(this.scene, this.camera);
			this.view.x = this.stage.stageWidth / 2;
			this.view.y = this.stage.stageHeight / 2;
			this.addChild(this.view);
			
			this.configureNodes();
			this.view.render();
		}
		
		/**
		 * Creates the Node graph
		 */
		private function configureNodes(): void
		{
			this.nodes = new Array();
			
			var i: uint;
			var j: uint;
			var k: uint;
			var node: Node3D;
			
			var w2: Number = Node3D.SIZE * this.xDepth / 2;
			var h2: Number = Node3D.SIZE * this.yDepth / 2;
			var d2: Number = Node3D.SIZE * this.zDepth / 2;
			
			for(i = 0; i < this.zDepth; i++)
			{
				this.nodes[i] = new Array();
				for(j = 0; j < this.yDepth; j++)
				{
					this.nodes[i][j] = new Array();
					for(k = 0; k < this.xDepth; k++)
					{
						node = new Node3D();
						
						node.x = k * Node3D.SIZE - w2;
						node.y = j * Node3D.SIZE - h2;
						node.z = i * Node3D.SIZE - d2;
						
						node.xPos = k;
						node.yPos = j;
						node.zPos = i;
						
						node.events.addEventListener(MouseEvent.MOUSE_DOWN, this.handleNodeMouseDown);
						
						this.yRotContainer.addChild(node);
						this.nodes[i][j][k] = node;
						
						if(j == 0) node.setWalkable(false); // create the initial floor
					}
				}
			}
		}
		
		// A*
		
		/**
		 * Starts the search for the path
		 */
		private function findPath(): void
		{
			this.path = new Array();
			this.open = new Array();
			this.closed = new Array();
			this.look(this.startNode);
		}
		
		/**
		 * Performs a search of adjacent nodes looking for the 
		 * most optimal path to the destination
		 */
		private function look(current: Node3D): void
		{
			if(current != null)
			{
				this.closed.push(current);
				
				if(current != this.destinationNode)
				{
					var startZ: uint = Math.max(current.zPos - 1, 0);
					var endZ: uint = Math.min(current.zPos + 1, this.zDepth - 1);
					
					var startY: uint = Math.max(current.yPos - 1, 0);
					var endY: uint = Math.min(current.yPos + 1, this.yDepth - 1);
					
					var startX: uint = Math.max(current.xPos - 1, 0);
					var endX: uint = Math.min(current.xPos + 1, this.zDepth - 1);
					
					var i: uint;
					var j: uint;
					var k: uint;
					var adjacentNode: Node3D;
					
					for(i = startZ; i <= endZ; i++)
					{
						for(j = startY; j <= endY; j++)
						{
							for(k = startX; k <= endX; k++)
							{
								if(!(i == current.zPos && j == current.yPos && k == current.xPos))
								{
									adjacentNode = this.nodes[i][j][k];
									this.searchNode(adjacentNode, current);
								}
							}
						}
					}
					
					this.open.sortOn("f", Array.NUMERIC);
					var best: Node3D = this.open.splice(0, 1)[0];
					this.look(best);
					
				}
				else
				{
					var s2: Number = Node3D.SIZE / 2; // the offset values will be used for animation
					var node: Node3D = this.destinationNode;
					while(node != this.startNode)
					{
						if(node != this.destinationNode) 
						{
							node.setPath(true);
							this.path.unshift({x: node.x + s2, y: node.y + s2, z: node.z + s2});
						}
						node = node.parentNode;
					}
					
					this.animate();
				}
			}
		}
		
		/**
		 * Searches a node and scores it based heuristics
		 */
		private function searchNode(node: Node3D, parent: Node3D): void
		{
			var xZDiagonal: Boolean = (node.xPos != parent.xPos && node.zPos != parent.zPos);
			var xYDiagonal: Boolean = (node.xPos != parent.xPos && node.yPos != parent.yPos);
			var yZDiagonal: Boolean = (node.yPos != parent.yPos && node.zPos != parent.zPos);

			// we can't cut solid corners so we first need to check for that
			var corner: Boolean = false;
			if(xZDiagonal || xYDiagonal || yZDiagonal) // we only cut a corner when moving diagonally
			{
				// Check the direction of travel
				var xDirection: int = (node.xPos - parent.xPos);
				var yDirection: int = (node.yPos - parent.yPos);
				var zDirection: int = (node.zPos - parent.zPos);
				
				var xZCorner1: Node3D = this.nodes[parent.zPos][parent.yPos][parent.xPos + xDirection];
				var xZCorner2: Node3D = this.nodes[parent.zPos + zDirection][parent.yPos][parent.xPos];
				
				var xYCorner1: Node3D = this.nodes[parent.zPos][parent.yPos][parent.xPos + xDirection];
				var xYCorner2: Node3D = this.nodes[parent.zPos][parent.yPos + yDirection][parent.xPos];
				
				var yZCorner1: Node3D = this.nodes[parent.zPos][parent.yPos + yDirection][parent.xPos];
				var yZCorner2: Node3D = this.nodes[parent.zPos + zDirection][parent.yPos][parent.xPos];
				
				corner = (xZCorner1 || xZCorner2 || xYCorner1 || xYCorner2 || yZCorner1 || yZCorner2);
			}
			
			if(node.getWalkable() && !this.hasItem(this.open, node) && !this.hasItem(this.closed, node) && !corner)
			{
				node.parentNode = parent;
				node.g = parent.g + (xZDiagonal || xYDiagonal || yZDiagonal) ? 14 : 10;
				
				var difX: Number = this.destinationNode.xPos - node.xPos;
				var difY: Number = this.destinationNode.yPos - node.yPos;
				var difZ: Number = this.destinationNode.zPos - node.zPos;
				
				if(difX < 0) difX = difX * -1;
				if(difY < 0) difY = difY * -1;
				if(difZ < 0) difZ = difZ * -1;

				node.h = (difX + difY + difZ) * 10;
				node.f = node.g + node.h;
				
				this.open.push(node);
			}
			else if(node.getWalkable() && !this.hasItem(this.closed, node) && !corner)
			{
				var g: Number = parent.g + (xZDiagonal || xYDiagonal || yZDiagonal) ? 14 : 10;
				if(g < node.g)
				{
					node.g = g;
					node.parentNode = parent;
				}
			}
		}
		
		/**
		 * Utility function go see if an Array has an item. Using Arrays
		 * is not the most efficient but it works for a proof of concept.
		 */
		private function hasItem(array: Array, item: Object): Boolean
		{
			var i: uint;
			var len: uint = array.length;
			for(i = 0; i < len; i++)
			{
				if(array[i] == item) return true;
			}
			return false;
		}
		
		//
		
		/**
		 * Create a sphere and animate it along the found path
		 */
		private function animate(): void
		{
			var s2: Number = Node3D.SIZE / 2;
			var x: Number = this.startNode.x + s2;
			var y: Number = this.startNode.y + s2;
			var z: Number = this.startNode.z + s2;
			
			var material: ShadingColorMaterial = new ShadingColorMaterial(0xbbbbbb, 0xffffff, 0xffffff);
			var sphere: Sphere = new Sphere(material, {x: x, y: y, z: z, radius: s2 * 0.75, segmentsW: 16, segmentsH: 12});
			this.yRotContainer.addChild(sphere);
			
			var tweenObj: Object = 
			{
				x: this.destinationNode.x + s2, 
				y: this.destinationNode.y + s2, 
				z: this.destinationNode.z + s2, 
				_bezier: this.path, 
				time: 5, 
				transition: "easeinoutquad",
				onUpdate: this.handleUpdate
			};
			
			//this.view.render();
			Tweener.addTween(sphere, tweenObj);
		}
		
		/**
		 * Handles call backs from the Tweener
		 */
		private function handleUpdate(): void
		{
			this.view.render();
		}
		
		// Node3D and link Event Handlers
		
		/**
		 * Handles mouse down events from the Nodes
		 */
		private function handleNodeMouseDown(event: MouseEvent3D): void
		{
			var node: Node3D = event.object.extra.node;
			
			if(this.mode == CLEAR_WALL)
			{
				if(node != this.startNode && node != this.destinationNode)
				{
					node.setWalkable(true);
					this.view.render();
				}
			}
			else
			{
				// Get the adjacent node
				var face: Face3D = (event.drawpri as DrawTriangle).face;
				
				var avgX: Number = (face.v0.x + face.v1.x + face.v2.x) / 3;
				var avgY: Number = (face.v0.y + face.v1.y + face.v2.y) / 3;
				var avgZ: Number = (face.v0.z + face.v1.z + face.v2.z) / 3;
				
				var s2: Number = Node3D.SIZE / 2;
				
				var xDir: Boolean = (avgX == s2 || avgX == -s2);
				var yDir: Boolean = (avgY == s2 || avgY == -s2);
				var zDir: Boolean = (avgZ == s2 || avgZ == -s2);

				var adjacentNode: Node3D;
				var xPos: int;
				var yPos: int;
				var zPos: int;
				
				if(xDir)
				{
					xPos = node.xPos + ((avgX > 0) ? 1 : -1);
					if(xPos >= 0 && xPos < this.xDepth)
					{
						yPos = node.yPos;
						zPos = node.zPos;
						adjacentNode = this.nodes[zPos][yPos][xPos];
					}
				}
				else if(yDir)
				{
					yPos = node.yPos + ((avgY > 0) ? 1 : -1);
					if(yPos >= 0 && yPos < this.yDepth)
					{
						xPos = node.xPos;
						zPos = node.zPos;
						adjacentNode = this.nodes[zPos][yPos][xPos];
					}
				}
				else if(zDir)
				{
					zPos = node.zPos + ((avgZ > 0) ? 1 : -1);
					if(zPos >= 0 && zPos < this.zDepth)
					{
						xPos = node.xPos;
						yPos = node.yPos;
						adjacentNode = this.nodes[zPos][yPos][xPos];
						
						trace(adjacentNode);
					}
				}
				
				if(adjacentNode != null)
				{
					switch(this.mode)
					{
						case CREATE_WALL:
							if(adjacentNode != this.startNode && adjacentNode != this.destinationNode)
							{
								adjacentNode.setWalkable(false);
							}
							break;
							
						case START:
							if(adjacentNode.getWalkable() && adjacentNode != this.destinationNode)
							{
								if(this.startNode != null) this.startNode.setStart(false);
								adjacentNode.setStart(true);
								this.startNode = adjacentNode;
							}
							break;
						
						case STOP:
							if(adjacentNode.getWalkable() && adjacentNode != this.startNode)
							{
								if(this.destinationNode != null) this.destinationNode.setDestination(false);
								adjacentNode.setDestination(true);
								this.destinationNode = adjacentNode;
							}
							break;
					}
				
					this.view.render();
				}
			}
		}
		
		/**
		 * Handles click events on the links
		 */
		private function handleFieldClick(event: MouseEvent): void
		{
			var field: TextField = event.target as TextField;
			
			if(field.text == "find path")
			{
				this.findPath();
			}
			else if(field.text == "clear")
			{
				this.startNode = null;
				this.destinationNode = null;
				this.removeChild(this.view);
				this.view = null;
				this.configureScene();
			}
			else
			{
				if(this.selectedField != null) this.selectedField.setTextFormat(new TextFormat(null, null, 0x3399ff));
				this.selectedField = field;
				this.selectedField.setTextFormat(new TextFormat(null, null, 0xff9933));
				
				switch(field.text)
				{
					case "create wall":
						this.mode = CREATE_WALL;
						break;
						
					case "clear wall":
						this.mode = CLEAR_WALL;
						break;
					
					case "start node":
						this.mode = START;
						break;
					
					case "stop node":
						this.mode = STOP;
						break;
				}
			}
		}
		
		// Stage Event Handlers
		
		/**
		 * Handles KeyDown events on the Stage
		 */
		private function handleKeyDown(event: KeyboardEvent): void
		{
			if(event.keyCode == Keyboard.DOWN)
			{
				this.rotatingX = -1;
			}
			else if(event.keyCode == Keyboard.UP)
			{
				this.rotatingX = 1;
			}
			
			if(event.keyCode == Keyboard.LEFT)
			{
				this.rotatingY = -1;
			}
			else if(event.keyCode == Keyboard.RIGHT)
			{
				this.rotatingY = 1;
			}
		}
		
		/**
		 * Handles KeyUp events on the Stage
		 */
		private function handleKeyUp(event: KeyboardEvent): void
		{
			if(event.keyCode == Keyboard.DOWN || event.keyCode == Keyboard.UP)
			{
				this.rotatingX = 0;
			}
			
			if(event.keyCode == Keyboard.LEFT || event.keyCode == Keyboard.RIGHT)
			{
				this.rotatingY = 0;
			}
		}
		
		/**
		 * Handles enter frame events
		 */
		private function handleEnterFrame(event: Event): void
		{
			if(this.rotatingX != 0)
			{
				this.xRotContainer.rotationX += this.rotatingX * 5;
			}
			if(this.rotatingY != 0)
			{
				this.yRotContainer.rotationY += this.rotatingY * 5;
			}
			
			if(this.rotatingX != 0 || this.rotatingY != 0) this.view.render();
		}
		
	}
	
}